% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/convert.R
\name{RMH_to_OR}
\alias{RMH_to_OR}
\alias{RMH_to_OR.default}
\alias{RMH_to_OR.data.frame}
\title{Convert Roe & Metz Parameters to Obuchowski-Rockette Parameters}
\usage{
RMH_to_OR(...)

\method{RMH_to_OR}{default}(
  ...,
  n0,
  n1,
  b,
  delta1,
  delta2,
  var_R,
  var_TR,
  var_C,
  var_TC,
  var_RC,
  var_error
)

\method{RMH_to_OR}{data.frame}(params, ...)
}
\arguments{
\item{...}{arguments passed to the default method.}

\item{n0, n1}{numbers of nondiseased and diseased cases.}

\item{b}{b>0, with 1/b^2 = ratio of each diseased-case variance component to
the corresponding diseased-case variance component. It follows that b is
also the conventional binormal-curve slope, i.e., the slope of each
reader's true ROC curve plotted in probit space.}

\item{delta1, delta2}{test 1 and test 2 separations of the RMH-model
nondiseased and diseased latent likelihood-of-disease rating distribution
means.}

\item{var_R, var_TR}{RMH-model reader and test-by-reader variance components.}

\item{var_C, var_TC, var_RC, var_error}{RMH-model case, test-by-case,
reader-by-case and error variance components for nondiseased cases.}

\item{params}{data frame of above RM parameter values in the columns.}
}
\value{
The OR model parameters are returned in a data frame with the following
elements.

\describe{
  \item{...}{arguments passed to the default method.}
  \item{AUC1, AUC2}{test 1 and 2 expected empirical AUCs.}
  \item{var_R, var_TR}{OR reader and test-by-reader variance components.}
  \item{corr1, corr2, corr3}{OR error correlations.}
  \item{var_error}{OR error variance.}
  \item{n0, n1}{number of nondiseased and diseased cases.}
}

Related quantities describing the true reader ROC curves that are also
returned in the data frame:

\describe{
  \item{b}{b > 0, with 1/b^2 = (RM diseased variance component) /
    (corresponding RM nondiseased variance component).}
  \item{mean_to_sig1}{expected mean-to-sigma ratio across readers for test
    1.}
  \item{mean_to_sig2}{expected mean-to-sigma ratio across readers for test
    2.}
  \item{Pr1_improper}{probability that the test 1 ROC curve for a random
    reader will be noticeably improper (i.e, |mean-to-sigma ratio| < 2).}
  \item{Pr2_improper}{probability that the test 2 ROC curve for a random
    reader will be noticeably improper (i.e, |mean-to-sigma ratio| < 2).}
}
}
\description{
Determines Obuchowski-Rockette (OR) model parameter values that describe the
distribution of  empirical AUC reader performance outcomes computed from
multireader multicase likelihood-of-disease rating data simulated from the
Roe & Metz (RM) simulation model, based on the analytical mapping provided by
Hillis (2018).  The function assumes the RM model proposed by (Hillis, 2012),
which generalizes the orginal RM model (Roe and Metz, 1997) by allowing the
latent confidence-of-disease rating distributions to have unequal
diseased-case and nondiseased-case variances, with the variance components
involving diseased cases constrained to differ by a factor of 1/b^2, b>0,
from corresponding variance components involving nondiseased cases.
\emph{Throughout we refer to the Hillis (2012) RM model as the
\bold{RMH model}.}
}
\details{
Hillis (2012) modified the original RM model (Roe and Metz, 1997) by allowing
variance components involving case to depend on truth (diseased/nondiseased),
with variance components involving diseased cases set equal to those
involving nondiseased cases multiplied by the factor 1/b^2, b>0. Assuming
this model, \emph{which we refer to as the \bold{RMH model}},
Hillis (2018) derived analytical formulas expressing OR
parameters that describe the distribution of empirical AUC outcomes computed
from RMH model simulated data as functions of the RMH parameters. This mapping
from the RMH parameters to the OR parameters is implemented in \R by the
RMH_to_OR function.

A related function is the OR_to_RMH function, which determines RM parameter
values corresponding to real-data or conjectured Obuchowski-Rockette (OR) parameter
estimates.
}
\examples{
##  Example 1: Computing OR parameters from RMH parameters directly
# RMH parameters from first line (A_z = 0.702) of Table 1 in Roe & Metz (1997)
# with 50 diseased and 50 nondiseased cases.
OR <- RMH_to_OR(n0 = 50, n1 = 50, delta1 = 0.75, delta2 = 0.75,
                var_R = 0.0055, var_TR = 0.0055, var_C = 0.3, var_TC = 0.3,
                var_RC = 0.2, var_error = 0.2, b = 1)
OR

##  Example 2: Computing OR parameters from a data frame of RMH parameters
##  ---------------------------------------------------------------------
## Example 2a:  RMH parameters from first line (A_z = 0.702) of Table 1 in
# Roe & Metz (1997) with 50 diseased and 50 nondiseased cases
RM_parms_line1 <- data.frame(n0 = 50, n1 = 50, delta1 = 0.75, delta2 = 0.75,
                             var_R = 0.0055, var_TR = 0.0055, var_C = 0.3, var_TC = 0.3,
                             var_RC = 0.2, var_error = 0.2, b = 1)
OR <- RMH_to_OR(RM_parms_line1)
OR
## Note below that applying the OR_to_RMH function to the above OR parameters
# results in the original RMH parameters within rounding error:
check <- OR_to_RMH(OR)
check

## Example 2b: RMH parameters from last 3 lines of Table 1 in Roe & Metz (1997)
# using 10 diseased and 25 nondiseased cases
RM_3_models <- data.frame(
  rbind(
    line6 = c(25, 10, 0.75, 0.75, 0.011, 0.011, 0.1, 0.1, 0.2, 0.6, 1),
    line7 = c(25, 10, 1.50, 1.50, 0.03, 0.03, 0.1, 0.1, 0.2, 0.6, 1),
    line8 = c(25, 10, 2.5, 2.5, 0.056, 0.056, 0.1, 0.1, 0.2, 0.6, 1)
  )
)
colnames(RM_3_models) <- c("n0", "n1", "delta1", "delta2", "var_R", "var_TR",
                           "var_C", "var_TC", "var_RC", "var_error", "b")
RM_3_models
OR_3_models <- RMH_to_OR(RM_3_models)
OR_3_models

## Example 2c: RMH parameters from last 3 lines of Table 1 in Hillis (2012)
# using 10 diseased and 25 nondiseased cases
RM_3_models_Hillis <- data.frame(
  rbind(
    line6 = c(25, 25, 0.821, 0.821, 0.0132, 0.0132, 0.1, 0.1, 0.2, 0.6, 0.84566),
    line7 = c(25, 25, 1.831, 1.831, 0.0447, 0.0447, 0.1, 0.1, 0.2, 0.6, 0.71082),
    line8 = c(25, 25, 3.661, 3.611, 0.1201, 0.1201, 0.1, 0.1, 0.2, 0.6, 0.55140)
  )
)
colnames(RM_3_models_Hillis) <- c("n0", "n1", "delta1", "delta2", "var_R", "var_TR",
                                  "var_C", "var_TC", "var_RC", "var_error", "b")
RM_3_models_Hillis
OR_3_models_Hillis <- RMH_to_OR(RM_3_models_Hillis)
OR_3_models_Hillis

}
\references{
Hillis SL (2012). Simulation of unequal-variance binormal multireader
ROC decision data: an extension of the Roe and Metz simulation model.
\emph{Academic Radiology}, 19(12): 1518-1528. doi: 10.1016/j.acra.2012.09.011

Hillis SL (2018). Relationship between Roe and Metz simulation model
for multireader diagnostic data and Obuchowski-Rockette model parameters.
\emph{Statistics in Medicine}, 37(13): 2067-2093. doi: 10.1002/sim.7616

Roe CA and Metz CE (1997). Dorfman-Berbaum-Metz method for statistical
analysis of multireader, multimodality receiver operating characteristic
data: validation with computer simulation. \emph{Academic Radiology}, 4(4):
298-303. doi: 10.1016/S1076-6332(97)80032-3
}
\seealso{
\code{\link{OR_to_RMH}}
}
\author{
Stephen L. Hillis, Departments of Radiology and Biostatistics,
University of Iowa, \email{steve-hillis@uiowa.edu}

Brian J. Smith, Department of Biostatistics, University of Iowa,
\email{brian-j-smith@uiowa.edu}
}
