#' Interconnection motif forms
#'
#' The dictionary of interconnection motif forms without intra-guild interactions.
#'
#' @encoding UTF-8
#'
#' @param ID Character. Default to "all". See details for the IDs representing different motifs.
#' @details
#' An interconnection motif is defined to comprise three sets of connected nodes: the connector nodes (belonging to b-nodes), the nodes in one subnetwork (belonging to a-nodes in the P subnetwork), and the nodes in the other subnetwork (belonging to c-nodes in the Q subnetwork). Each motif has maximumly 6 nodes, resulting in a total of 48 distinct motif forms.
#'
#' @return
#'
#' If \code{ID} is one ID, it returns a corresponding motif "igraph" object.
#'
#' If \code{ID} is "all", it returns a list containing all 48 motifs of "igraph" objects.
#'
#' @export
#'
#' @importFrom igraph V
#' @importFrom igraph as.directed
#' @importFrom igraph graph_from_edgelist
#' @importFrom igraph layout_with_sugiyama
#' @examples
#'
#' ID<-c("M111","M112","M113","M114","M211","M212","M213","M311",
#' "M312","M411","M121","M122-1","M122-2","M122-3","M123-1",
#' "M123-2","M123-3","M123-4","M123-5","M221-1","M221-2",
#' "M221-3","M222-1","M222-2","M222-3","M222-4","M222-5",
#' "M222-6","M222-7","M222-8","M222-9","M321-1","M321-2",
#' "M321-3","M321-4","M321-5","M131","M132-1","M132-2",
#' "M132-3","M132-4","M132-5","M231-1","M231-2","M231-3",
#' "M231-4","M231-5","M141")
#'
#' m<-Multi_motif("M222-4")
#' plot(m,main="M222-4")
#'
#' mr <- par(mfrow=c(4,6),mar=c(1,1,3,1))
#' for(i in 1:24){
#'     plot(Multi_motif("all")[[i]],
#'     vertex.size=30, vertex.label=NA,
#'     vertex.color="blue",main=ID[i])
#' }
#' par(mr)
#'

 Multi_motif<-function(ID=c("M111","M112","M113","M114","M211","M212","M213","M311","M312","M411","M121","M122-1",
                              "M122-2","M122-3","M123-1","M123-2","M123-3","M123-4","M123-5","M221-1","M221-2",
                              "M221-3","M222-1","M222-2","M222-3","M222-4","M222-5","M222-6","M222-7","M222-8",
                              "M222-9","M321-1","M321-2","M321-3","M321-4","M321-5","M131","M132-1","M132-2",
                              "M132-3","M132-4","M132-5","M231-1","M231-2","M231-3","M231-4","M231-5","M141",
                              "all")){
    if(is.null(ID))
        ID<-"all"
    if(ID=="M111"){
        M111<-matrix(c("1","2","2","3"),2,2)
        M111<-graph_from_edgelist(M111)
        M111<-as.undirected(M111)
        V(M111)$level<-0:2
        M111$layout<-c(M111,layers=V(M111)$level)$layout
       return(M111)
    }
    if(ID=="M112"){
       M112<-matrix(c("1","2","2","2","3","4"),3,2)
       M112<-graph_from_edgelist(M112)
       M112<-as.undirected(M112)
       V(M112)$level<-c(0:2,2)
       M112$layout<-layout_with_sugiyama(M112,layers=V(M112)$level)$layout
       return(M112)
    }
    if(ID=="M113"){
       M113<-matrix(c("1","2","2","2","2","3","4","5"),4,2)
       M113<-graph_from_edgelist(M113)
       M113<-as.undirected(M113)
       V(M113)$level<-c(0:2,2,2)
       M113$layout<-layout_with_sugiyama(M113,layers=V(M113)$level)$layout
       return(M113)
    }
    if(ID=="M114"){
        M114<-matrix(c("1","2","2","2","2","2","3","4","5","6"),5,2)
        M114<-graph_from_edgelist(M114)
        M114<-as.undirected(M114)
        V(M114)$level<-c(0:2,2,2,2)
        M114$layout<-layout_with_sugiyama(M114,layers=V(M114)$level)$layout
        return(M114)
    }
    if(ID=="M211"){
       M211<-matrix(c("1","2","3","3","3","4"),3,2)
       M211<-graph_from_edgelist(M211)
       NET<-as.undirected(M211)
       V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3")]<-1
       V(NET)$level[(V(NET)$name)%in%c("4")]<-2
       M211<-NET
       M211$layout<-layout_with_sugiyama(M211,layers=V(M211)$level)$layout
       return(M211)
    }
    if(ID=="M212"){
       M212<-matrix(c("1","2","3","3","3","3","4","5"),4,2)
       M212<-graph_from_edgelist(M212)
       NET<-as.undirected(M212)
       V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3")]<-1
       V(NET)$level[(V(NET)$name)%in%c("5","4")]<-2
       M212<-NET
       M212$layout<-layout_with_sugiyama(M212,layers=V(M212)$level)$layout
       return(M212)
    }
    if(ID=="M213"){
       M213<-matrix(c("1","2","3","3","3","3","3","4","5","6"),5,2)
       M213<-graph_from_edgelist(M213)
       NET<-as.undirected(M213)
       V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3")]<-1
       V(NET)$level[(V(NET)$name)%in%c("4","5","6")]<-2
       M213<-NET
       M213$layout<-layout_with_sugiyama(M213,layers=V(M213)$level)$layout
       return(M213)
    }
    if(ID=="M311"){
       M311<-matrix(c("1","2","3","4","4","4","4","5"),4,2)
       M311<-graph_from_edgelist(M311)
       NET<-as.undirected(M311)
       V(NET)$level[(V(NET)$name)%in%c("1","2","3")]<-0
       V(NET)$level[(V(NET)$name)%in%c("4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("5")]<-2
       M311<-NET
       M311$layout<-layout_with_sugiyama(M311,layers=V(M311)$level)$layout
       return(M311)
    }
    if(ID=="M312"){
       M312<-matrix(c("1","2","3","4","4","4","4","4","5","6"),5,2)
       M312<-graph_from_edgelist(M312)
       NET<-as.undirected(M312)
       V(NET)$level[(V(NET)$name)%in%c("1","2","3")]<-0
       V(NET)$level[(V(NET)$name)%in%c("4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("5","6")]<-2
       M312<-NET
       M312$layout<-layout_with_sugiyama(M312,layers=V(M312)$level)$layout
       return(M312)
    }
    if(ID=="M411"){
        M411<-matrix(c("1","2","3","4","5","5","5","5","5","6"),5,2)
        M411<-graph_from_edgelist(M411)
        M411<-as.undirected(M411)
        V(M411)$level<-c(0,1,0,0,0,2)
        M411$layout<-layout_with_sugiyama(M411,layers=V(M411)$level)$layout
        return(M411)
    }
    if(ID=="M121"){
       M1211<-matrix(c("1","1","2","3","2","3","4","4"),4,2)
       M1211<-graph_from_edgelist(M1211)
       NET<-as.undirected(M1211)
       V(NET)$level[(V(NET)$name)%in%c("1")]<-0
       V(NET)$level[(V(NET)$name)%in%c("2","3")]<-1
       V(NET)$level[(V(NET)$name)%in%c("4")]<-2
       M1211<-NET
       M1211$layout<-layout_with_sugiyama(M1211,layers=V(M1211)$level)$layout
       return(M1211)
    }
    if(ID=="M122-1"){
       M1221<-matrix(c("1","2","1","3","2","4","3","5"),4,2)
       M1221<-graph_from_edgelist(M1221)
       NET<-as.undirected(M1221)
       V(NET)$level[(V(NET)$name)%in%c("1")]<-0
       V(NET)$level[(V(NET)$name)%in%c("2","3")]<-1
       V(NET)$level[(V(NET)$name)%in%c("4","5")]<-2
       M1221<-NET
       M1221$layout<-layout_with_sugiyama(M1221,layers=V(M1221)$level)$layout
       return(M1221)
    }
    if(ID=="M122-2"){
       M1222<-matrix(c("1","1","2","3","3","2","3","4","4","5"),5,2)
       M1222<-graph_from_edgelist(M1222)
       NET<-as.undirected(M1222)
       V(NET)$level[(V(NET)$name)%in%c("1")]<-0
       V(NET)$level[(V(NET)$name)%in%c("2","3")]<-1
       V(NET)$level[(V(NET)$name)%in%c("4","5")]<-2
       M1222<-NET
       M1222$layout<-layout_with_sugiyama(M1222,layers=V(M1222)$level)$layout
       return(M1222)
    }
    if(ID=="M122-3"){
       M1223<-matrix(c("1","1","2","2","3","3","2","3","4","5","4","5"),6,2)
       M1223<-graph_from_edgelist(M1223)
       NET<-as.undirected(M1223)
       V(NET)$level[(V(NET)$name)%in%c("1")]<-0
       V(NET)$level[(V(NET)$name)%in%c("2","3")]<-1
       V(NET)$level[(V(NET)$name)%in%c("4","5")]<-2
       M1223<-NET
       M1223$layout<-layout_with_sugiyama(M1223,layers=V(M1223)$level)$layout
       return(M1223)
    }
    if(ID=="M123-1"){
       M1231<-matrix(c("1","1","2","2","3","2","3","4","5","6"),5,2)
       M1231<-graph_from_edgelist(M1231)
       NET<-as.undirected(M1231)
       V(NET)$level[(V(NET)$name)%in%c("1")]<-0
       V(NET)$level[(V(NET)$name)%in%c("2","3")]<-1
       V(NET)$level[(V(NET)$name)%in%c("4","5","6")]<-2
       M1231<-NET
       M1231$layout<-layout_with_sugiyama(M1231,layers=V(M1231)$level)$layout
       return(M1231)
    }
    if(ID=="M123-2"){
       M1232<-matrix(c("1","1","2","3","3","3","2","3","4","4","5","6"),6,2)
       M1232<-graph_from_edgelist(M1232)
       NET<-as.undirected(M1232)
       V(NET)$level[(V(NET)$name)%in%c("1")]<-0
       V(NET)$level[(V(NET)$name)%in%c("2","3")]<-1
       V(NET)$level[(V(NET)$name)%in%c("4","5","6")]<-2
       M1232<-NET
       M1232$layout<-layout_with_sugiyama(M1232,layers=V(M1232)$level)$layout
       return(M1232)
    }
    if(ID=="M123-3"){
       M1233<-matrix(c("1","1","2","2","3","3","2","3","4","5","5","6"),6,2)
       M1233<-graph_from_edgelist(M1233)
       NET<-as.undirected(M1233)
       V(NET)$level[(V(NET)$name)%in%c("1")]<-0
       V(NET)$level[(V(NET)$name)%in%c("2","3")]<-1
       V(NET)$level[(V(NET)$name)%in%c("4","5","6")]<-2
       M1233<-NET
       M1233$layout<-layout_with_sugiyama(M1233,layers=V(M1233)$level)$layout
       return(M1233)
    }
    if(ID=="M123-4"){
       M1234<-matrix(c("1","1","2","2","2","3","3","2","3","4","5","6","5","6"),7,2)
       M1234<-graph_from_edgelist(M1234)
       NET<-as.undirected(M1234)
       V(NET)$level[(V(NET)$name)%in%c("1")]<-0
       V(NET)$level[(V(NET)$name)%in%c("2","3")]<-1
       V(NET)$level[(V(NET)$name)%in%c("4","5","6")]<-2
       M1234<-NET
       M1234$layout<-layout_with_sugiyama(M1234,layers=V(M1234)$level)$layout
       return(M1234)
    }
    if(ID=="M123-5"){
       M1235<-matrix(c("1","1","2","2","2","3","3","3","2","3","4","5","6","4","5","6"),8,2)
       M1235<-graph_from_edgelist(M1235)
       NET<-as.undirected(M1235)
       V(NET)$level[(V(NET)$name)%in%c("1")]<-0
       V(NET)$level[(V(NET)$name)%in%c("2","3")]<-1
       V(NET)$level[(V(NET)$name)%in%c("4","5","6")]<-2
       M1235<-NET
       M1235$layout<-layout_with_sugiyama(M1235,layers=V(M1235)$level)$layout
       return(M1235)
    }
    if(ID=="M221-1"){
       M2211<-matrix(c("1","2","3","4","3","4","5","5"),4,2)
       M2211<-graph_from_edgelist(M2211)
       NET<-as.undirected(M2211)
       V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("5")]<-2
       M2211<-NET
       M2211$layout<-layout_with_sugiyama(M2211,layers=V(M2211)$level)$layout
       return(M2211)
    }
    if(ID=="M221-2"){
       M2212<-matrix(c("1","2","2","3","4","3","3","4","5","5"),5,2)
       M2212<-graph_from_edgelist(M2212)
       NET<-as.undirected(M2212)
       V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("5")]<-2
       M2212<-NET
       M2212$layout<-layout_with_sugiyama(M2212,layers=V(M2212)$level)$layout
       return(M2212)
    }
    if(ID=="M221-3"){
       M2213<-matrix(c("1","1","2","2","3","4","3","4","3","4","5","5"),6,2)
       M2213<-graph_from_edgelist(M2213)
       NET<-as.undirected(M2213)
       V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("5")]<-2
       M2213<-NET
       M2213$layout<-layout_with_sugiyama(M2213,layers=V(M2213)$level)$layout
       return(M2213)
    }
    if(ID=="M222-1"){
       M2221<-matrix(c("1","2","2","3","4","3","3","4","5","6"),5,2)
       M2221<-graph_from_edgelist(M2221)
       NET<-as.undirected(M2221)
       V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("5","6")]<-2
       M2221<-NET
       M2221$layout<-layout_with_sugiyama(M2221,layers=V(M2221)$level)$layout
       return(M2221)
    }
    if(ID=="M222-2"){
       M2222<-matrix(c("1","1","2","2","3","4","3","4","3","4","5","6"),6,2)
       M2222<-graph_from_edgelist(M2222)
       NET<-as.undirected(M2222)
       V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("5","6")]<-2
       M2222<-NET
       M2222$layout<-layout_with_sugiyama(M2222,layers=V(M2222)$level)$layout

       return(M2222)
    }
    if(ID=="M222-3"){
       M2223<-matrix(c("1","2","3","3","4","3","4","5","6","6"),5,2)
       M2223<-graph_from_edgelist(M2223)
       NET<-as.undirected(M2223)
       V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("5","6")]<-2
       M2223<-NET
       M2223$layout<-layout_with_sugiyama(M2223,layers=V(M2223)$level)$layout
       return(M2223)
    }
    if(ID=="M222-4"){
       M2224<-matrix(c("1","2","2","3","3","4","3","3","4","5","6","6"),6,2)
       M2224<-graph_from_edgelist(M2224)
       NET<-as.undirected(M2224)
       V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("5","6")]<-2
       M2224<-NET
       M2224$layout<-layout_with_sugiyama(M2224,layers=V(M2224)$level)$layout
       return(M2224)
    }
    if(ID=="M222-5"){
       M2225<-matrix(c("1","1","2","2","3","4","4","3","4","3","4","5","5","6"),7,2)
       M2225<-graph_from_edgelist(M2225)
       NET<-as.undirected(M2225)
       V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("5","6")]<-2
       M2225<-NET
       M2225$layout<-layout_with_sugiyama(M2225,layers=V(M2225)$level)$layout
       return(M2225)
    }
    if(ID=="M222-6"){
       M2226<-matrix(c("1","2","3","3","4","4","3","4","5","6","5","6"),6,2)
       M2226<-graph_from_edgelist(M2226)
       NET<-as.undirected(M2226)
       V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("5","6")]<-2
       M2226<-NET
       M2226$layout<-layout_with_sugiyama(M2226,layers=V(M2226)$level)$layout
       return(M2226)
    }
    if(ID=="M222-7"){
       M2227<-matrix(c("1","2","2","3","3","4","4","3","3","4","5","6","5","6"),7,2)
       M2227<-graph_from_edgelist(M2227)
       NET<-as.undirected(M2227)
       V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("5","6")]<-2
       M2227<-NET
       M2227$layout<-layout_with_sugiyama(M2227,layers=V(M2227)$level)$layout
       return(M2227)
    }
    if(ID=="M222-8"){
       M2228<-matrix(c("1","1","2","2","3","3","4","4","3","4","3","4","5","6","5","6"),8,2)
       M2228<-graph_from_edgelist(M2228)
       NET<-as.undirected(M2228)
       V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("5","6")]<-2
       M2228<-NET
       M2228$layout<-layout_with_sugiyama(M2228,layers=V(M2228)$level)$layout
       return(M2228)
    }
    if(ID=="M222-9"){
        M2229<-matrix(c("1","2","2","3","4","4","3","3","4","5","5","6"),6,2)
        M2229<-graph_from_edgelist(M2229)
        NET<-as.undirected(M2229)
        V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
        V(NET)$level[(V(NET)$name)%in%c("3","4")]<-1
        V(NET)$level[(V(NET)$name)%in%c("5","6")]<-2
        M2229<-NET
        M2229$layout<-layout_with_sugiyama(M2229,layers=V(M2229)$level)$layout
        return(M2229)
    }
    if(ID=="M321-1"){
       M3211<-matrix(c("1","2","3","4","5","4","4","5","6","6"),5,2)
       M3211<-graph_from_edgelist(M3211)
       NET<-as.undirected(M3211)
       V(NET)$level[(V(NET)$name)%in%c("1","2","3")]<-0
       V(NET)$level[(V(NET)$name)%in%c("5","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("6")]<-2
       M3211<-NET
       M3211$layout<-layout_with_sugiyama(M3211,layers=V(M3211)$level)$layout
       return(M3211)
    }
    if(ID=="M321-2"){
       M3212<-matrix(c("1","1","2","3","4","5","4","5","5","5","6","6"),6,2)
       M3212<-graph_from_edgelist(M3212)
       NET<-as.undirected(M3212)
       V(NET)$level[(V(NET)$name)%in%c("1","2","3")]<-0
       V(NET)$level[(V(NET)$name)%in%c("5","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("6")]<-2
       M3212<-NET
       M3212$layout<-layout_with_sugiyama(M3212,layers=V(M3212)$level)$layout
       return(M3212)
    }
    if(ID=="M321-3"){
       M3213<-matrix(c("1","2","2","3","4","5","4","4","5","5","6","6"),6,2)
       M3213<-graph_from_edgelist(M3213)
       NET<-as.undirected(M3213)
       V(NET)$level[(V(NET)$name)%in%c("1","2","3")]<-0
       V(NET)$level[(V(NET)$name)%in%c("5","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("6")]<-2
       M3213<-NET
       M3213$layout<-layout_with_sugiyama(M3213,layers=V(M3213)$level)$layout
       return(M3213)
    }
    if(ID=="M321-4"){
       M3214<-matrix(c("1","1","2","2","3","4","5","4","5","4","5","5","6","6"),7,2)
       M3214<-graph_from_edgelist(M3214)
       NET<-as.undirected(M3214)
       V(NET)$level[(V(NET)$name)%in%c("1","2","3")]<-0
       V(NET)$level[(V(NET)$name)%in%c("5","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("6")]<-2
       M3214<-NET
       M3214$layout<-layout_with_sugiyama(M3214,layers=V(M3214)$level)$layout
       return(M3214)
    }
    if(ID=="M321-5"){
       M3215<-matrix(c("1","1","2","2","3","3","4","5","4","5","4","5","4","5","6","6"),8,2)
       M3215<-graph_from_edgelist(M3215)
       NET<-as.undirected(M3215)
       V(NET)$level[(V(NET)$name)%in%c("1","2","3")]<-0
       V(NET)$level[(V(NET)$name)%in%c("5","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("6")]<-2
       M3215<-NET
       M3215$layout<-layout_with_sugiyama(M3215,layers=V(M3215)$level)$layout
       return(M3215)
    }
    if(ID=="M131"){
       M131<-matrix(c("1","1","1","2","3","4","2","3","4","5","5","5"),6,2)
       M131<-graph_from_edgelist(M131)
       NET<-as.undirected(M131)
       V(NET)$level[(V(NET)$name)%in%c("1")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3","2","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("5")]<-2
       M131<-NET
       M131$layout<-layout_with_sugiyama(M131,layers=V(M131)$level)$layout
       return(M131)
    }
    if(ID=="M132-1"){
       M1321<-matrix(c("1","1","1","2","3","4","2","3","4","5","5","6"),6,2)
       M1321<-graph_from_edgelist(M1321)
       NET<-as.undirected(M1321)
       V(NET)$level[(V(NET)$name)%in%c("1")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3","2","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("6","5")]<-2
       M1321<-NET
       M1321$layout<-layout_with_sugiyama(M1321,layers=V(M1321)$level)$layout
       return(M1321)
    }
    if(ID=="M132-2"){
       M1322<-matrix(c("1","1","1","2","2","3","4","2","3","4","5","6","6","6"),7,2)
       M1322<-graph_from_edgelist(M1322)
       NET<-as.undirected(M1322)
       V(NET)$level[(V(NET)$name)%in%c("1")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3","2","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("6","5")]<-2
       M1322<-NET
       M1322$layout<-layout_with_sugiyama(M1322,layers=V(M1322)$level)$layout
       return(M1322)
    }
    if(ID=="M132-3"){
       M1323<-matrix(c("1","1","1","2","3","3","4","2","3","4","5","5","6","6"),7,2)
       M1323<-graph_from_edgelist(M1323)
       NET<-as.undirected(M1323)
       V(NET)$level[(V(NET)$name)%in%c("1")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3","2","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("6","5")]<-2
       M1323<-NET
       M1323$layout<-layout_with_sugiyama(M1323,layers=V(M1323)$level)$layout
       return(M1323)
    }
    if(ID=="M132-4"){
       M1324<-matrix(c("1","1","1","2","2","3","3","4","2","3","4","5","6","5","6","6"),8,2)
       M1324<-graph_from_edgelist(M1324)
       NET<-as.undirected(M1324)
       V(NET)$level[(V(NET)$name)%in%c("1")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3","2","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("6","5")]<-2
       M1324<-NET
       M1324$layout<-layout_with_sugiyama(M1324,layers=V(M1324)$level)$layout
       return(M1324)
    }
    if(ID=="M132-5"){
       M1325<-matrix(c("1","1","1","2","2","3","3","4","4","2","3","4","5","6","5","6","5","6"),9,2)
       M1325<-graph_from_edgelist(M1325)
       NET<-as.undirected(M1325)
       V(NET)$level[(V(NET)$name)%in%c("1")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3","2","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("6","5")]<-2
       M1325<-NET
       M1325$layout<-layout_with_sugiyama(M1325,layers=V(M1325)$level)$layout
       return(M1325)
    }
    if(ID=="M231-1"){
       M2311<-matrix(c("1","1","2","3","4","5","3","4","5","6","6","6"),6,2)
       M2311<-graph_from_edgelist(M2311)
       NET<-as.undirected(M2311)
       V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
       V(NET)$level[(V(NET)$name)%in%c("5","3","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("6")]<-2
       M2311<-NET
       M2311$layout<-layout_with_sugiyama(M2311,layers=V(M2311)$level)$layout
       return(M2311)
    }
    if(ID=="M231-2"){
       M2312<-matrix(c("1","2","2","2","3","4","5","3","3","4","5","6","6","6"),7,2)
       M2312<-graph_from_edgelist(M2312)
       NET<-as.undirected(M2312)
       V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
       V(NET)$level[(V(NET)$name)%in%c("5","3","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("6")]<-2
       M2312<-NET
       M2312$layout<-layout_with_sugiyama(M2312,layers=V(M2312)$level)$layout
       return(M2312)
    }
    if(ID=="M231-3"){
       M2313<-matrix(c("1","1","2","2","3","4","5","3","4","4","5","6","6","6"),7,2)
       M2313<-graph_from_edgelist(M2313)
       NET<-as.undirected(M2313)
       V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
       V(NET)$level[(V(NET)$name)%in%c("5","3","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("6")]<-2
       M2313<-NET
       M2313$layout<-layout_with_sugiyama(M2313,layers=V(M2313)$level)$layout
       return(M2313)
    }
    if(ID=="M231-4"){
       M2314<-matrix(c("1","1","1","2","2","3","4","5","3","4","5","4","5","6","6","6"),8,2)
       M2314<-graph_from_edgelist(M2314)
       NET<-as.undirected(M2314)
       V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
       V(NET)$level[(V(NET)$name)%in%c("5","3","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("6")]<-2
       M2314<-NET
       M2314$layout<-layout_with_sugiyama(M2314,layers=V(M2314)$level)$layout
       return(M2314)
    }
    if(ID=="M231-5"){
       M2315<-matrix(c("1","1","1","2","2","2","3","4","5","3","4","5","3","4","5","6","6","6"),9,2)
       M2315<-graph_from_edgelist(M2315)
       NET<-as.undirected(M2315)
       V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
       V(NET)$level[(V(NET)$name)%in%c("5","3","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("6")]<-2
       M2315<-NET
       M2315$layout<-layout_with_sugiyama(M2315,layers=V(M2315)$level)$layout
       return(M2315)
    }
    if(ID=="M141"){
        M141<-matrix(c("1","1","1","1","2","3","4","5","2","3","4","5","6","6","6","6"),8,2)
        M141<-graph_from_edgelist(M141)
        NET<-as.undirected(M141)
        V(NET)$level[(V(NET)$name)%in%c("1")]<-0
        V(NET)$level[(V(NET)$name)%in%c("2","3","4","5")]<-1
        V(NET)$level[(V(NET)$name)%in%c("6")]<-2
        M141<-NET
        M141$layout<-layout_with_sugiyama(M141,layers=V(M141)$level)$layout
        return(M141)
    }

    if(ID=="all"){
       M111<-matrix(c("1","2","2","3"),2,2)
       M111<-graph_from_edgelist(M111)
       M111<-as.undirected(M111)
       V(M111)$level<-0:2
       M111$layout<-layout_with_sugiyama(M111,layers=V(M111)$level)$layout

       M112<-matrix(c("1","2","2","2","3","4"),3,2)
       M112<-graph_from_edgelist(M112)
       M112<-as.undirected(M112)
       V(M112)$level<-c(0:2,2)
       M112$layout<-layout_with_sugiyama(M112,layers=V(M112)$level)$layout

       M113<-matrix(c("1","2","2","2","2","3","4","5"),4,2)
       M113<-graph_from_edgelist(M113)
       M113<-as.undirected(M113)
       V(M113)$level<-c(0:2,2,2)
       M113$layout<-layout_with_sugiyama(M113,layers=V(M113)$level)$layout

       M114<-matrix(c("1","2","2","2","2","2","3","4","5","6"),5,2)
       M114<-graph_from_edgelist(M114)
       M114<-as.undirected(M114)
       V(M114)$level<-c(0:2,2,2,2)
       M114$layout<-layout_with_sugiyama(M114,layers=V(M114)$level)$layout

       M211<-matrix(c("1","2","3","3","3","4"),3,2)
       M211<-graph_from_edgelist(M211)
       NET<-as.undirected(M211)
       V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3")]<-1
       V(NET)$level[(V(NET)$name)%in%c("4")]<-2
       M211<-NET
       M211$layout<-layout_with_sugiyama(M211,layers=V(M211)$level)$layout

       M212<-matrix(c("1","2","3","3","3","3","4","5"),4,2)
       M212<-graph_from_edgelist(M212)
       NET<-as.undirected(M212)
       V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3")]<-1
       V(NET)$level[(V(NET)$name)%in%c("5","4")]<-2
       M212<-NET
       M212$layout<-layout_with_sugiyama(M212,layers=V(M212)$level)$layout

       M213<-matrix(c("1","2","3","3","3","3","3","4","5","6"),5,2)
       M213<-graph_from_edgelist(M213)
       NET<-as.undirected(M213)
       V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3")]<-1
       V(NET)$level[(V(NET)$name)%in%c("4","5","6")]<-2
       M213<-NET
       M213$layout<-layout_with_sugiyama(M213,layers=V(M213)$level)$layout

       M311<-matrix(c("1","2","3","4","4","4","4","5"),4,2)
       M311<-graph_from_edgelist(M311)
       NET<-as.undirected(M311)
       V(NET)$level[(V(NET)$name)%in%c("1","2","3")]<-0
       V(NET)$level[(V(NET)$name)%in%c("4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("5")]<-2
       M311<-NET
       M311$layout<-layout_with_sugiyama(M311,layers=V(M311)$level)$layout

       M312<-matrix(c("1","2","3","4","4","4","4","4","5","6"),5,2)
       M312<-graph_from_edgelist(M312)
       NET<-as.undirected(M312)
       V(NET)$level[(V(NET)$name)%in%c("1","2","3")]<-0
       V(NET)$level[(V(NET)$name)%in%c("4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("5","6")]<-2
       M312<-NET
       M312$layout<-layout_with_sugiyama(M312,layers=V(M312)$level)$layout

       M411<-matrix(c("1","2","3","4","5","5","5","5","5","6"),5,2)
       M411<-graph_from_edgelist(M411)
       M411<-as.undirected(M411)
       V(M411)$level<-c(0,1,0,0,0,2)
       M411$layout<-layout_with_sugiyama(M411,layers=V(M411)$level)$layout

       M1211<-matrix(c("1","1","2","3","2","3","4","4"),4,2)
       M1211<-graph_from_edgelist(M1211)
       NET<-as.undirected(M1211)
       V(NET)$level[(V(NET)$name)%in%c("1")]<-0
       V(NET)$level[(V(NET)$name)%in%c("2","3")]<-1
       V(NET)$level[(V(NET)$name)%in%c("4")]<-2
       M1211<-NET
       M1211$layout<-layout_with_sugiyama(M1211,layers=V(M1211)$level)$layout

       M1221<-matrix(c("1","2","1","3","2","4","3","5"),4,2)
       M1221<-graph_from_edgelist(M1221)
       NET<-as.undirected(M1221)
       V(NET)$level[(V(NET)$name)%in%c("1")]<-0
       V(NET)$level[(V(NET)$name)%in%c("2","3")]<-1
       V(NET)$level[(V(NET)$name)%in%c("4","5")]<-2
       M1221<-NET
       M1221$layout<-layout_with_sugiyama(M1221,layers=V(M1221)$level)$layout

       M1222<-matrix(c("1","1","2","3","3","2","3","4","4","5"),5,2)
       M1222<-graph_from_edgelist(M1222)
       NET<-as.undirected(M1222)
       V(NET)$level[(V(NET)$name)%in%c("1")]<-0
       V(NET)$level[(V(NET)$name)%in%c("2","3")]<-1
       V(NET)$level[(V(NET)$name)%in%c("4","5")]<-2
       M1222<-NET
       M1222$layout<-layout_with_sugiyama(M1222,layers=V(M1222)$level)$layout

       M1223<-matrix(c("1","1","2","2","3","3","2","3","4","5","4","5"),6,2)
       M1223<-graph_from_edgelist(M1223)
       NET<-as.undirected(M1223)
       V(NET)$level[(V(NET)$name)%in%c("1")]<-0
       V(NET)$level[(V(NET)$name)%in%c("2","3")]<-1
       V(NET)$level[(V(NET)$name)%in%c("4","5")]<-2
       M1223<-NET
       M1223$layout<-layout_with_sugiyama(M1223,layers=V(M1223)$level)$layout

       M1231<-matrix(c("1","1","2","2","3","2","3","4","5","6"),5,2)
       M1231<-graph_from_edgelist(M1231)
       NET<-as.undirected(M1231)
       V(NET)$level[(V(NET)$name)%in%c("1")]<-0
       V(NET)$level[(V(NET)$name)%in%c("2","3")]<-1
       V(NET)$level[(V(NET)$name)%in%c("4","5","6")]<-2
       M1231<-NET
       M1231$layout<-layout_with_sugiyama(M1231,layers=V(M1231)$level)$layout

       M1232<-matrix(c("1","1","2","3","3","3","2","3","4","4","5","6"),6,2)
       M1232<-graph_from_edgelist(M1232)
       NET<-as.undirected(M1232)
       V(NET)$level[(V(NET)$name)%in%c("1")]<-0
       V(NET)$level[(V(NET)$name)%in%c("2","3")]<-1
       V(NET)$level[(V(NET)$name)%in%c("4","5","6")]<-2
       M1232<-NET
       M1232$layout<-layout_with_sugiyama(M1232,layers=V(M1232)$level)$layout

       M1233<-matrix(c("1","1","2","2","3","3","2","3","4","5","5","6"),6,2)
       M1233<-graph_from_edgelist(M1233)
       NET<-as.undirected(M1233)
       V(NET)$level[(V(NET)$name)%in%c("1")]<-0
       V(NET)$level[(V(NET)$name)%in%c("2","3")]<-1
       V(NET)$level[(V(NET)$name)%in%c("4","5","6")]<-2
       M1233<-NET
       M1233$layout<-layout_with_sugiyama(M1233,layers=V(M1233)$level)$layout

       M1234<-matrix(c("1","1","2","2","2","3","3","2","3","4","5","6","5","6"),7,2)
       M1234<-graph_from_edgelist(M1234)
       NET<-as.undirected(M1234)
       V(NET)$level[(V(NET)$name)%in%c("1")]<-0
       V(NET)$level[(V(NET)$name)%in%c("2","3")]<-1
       V(NET)$level[(V(NET)$name)%in%c("4","5","6")]<-2
       M1234<-NET
       M1234$layout<-layout_with_sugiyama(M1234,layers=V(M1234)$level)$layout

       M1235<-matrix(c("1","1","2","2","2","3","3","3","2","3","4","5","6","4","5","6"),8,2)
       M1235<-graph_from_edgelist(M1235)
       NET<-as.undirected(M1235)
       V(NET)$level[(V(NET)$name)%in%c("1")]<-0
       V(NET)$level[(V(NET)$name)%in%c("2","3")]<-1
       V(NET)$level[(V(NET)$name)%in%c("4","5","6")]<-2
       M1235<-NET
       M1235$layout<-layout_with_sugiyama(M1235,layers=V(M1235)$level)$layout

       M2211<-matrix(c("1","2","3","4","3","4","5","5"),4,2)
       M2211<-graph_from_edgelist(M2211)
       NET<-as.undirected(M2211)
       V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("5")]<-2
       M2211<-NET
       M2211$layout<-layout_with_sugiyama(M2211,layers=V(M2211)$level)$layout

       M2212<-matrix(c("1","2","2","3","4","3","3","4","5","5"),5,2)
       M2212<-graph_from_edgelist(M2212)
       NET<-as.undirected(M2212)
       V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("5")]<-2
       M2212<-NET
       M2212$layout<-layout_with_sugiyama(M2212,layers=V(M2212)$level)$layout

       M2213<-matrix(c("1","1","2","2","3","4","3","4","3","4","5","5"),6,2)
       M2213<-graph_from_edgelist(M2213)
       NET<-as.undirected(M2213)
       V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("5")]<-2
       M2213<-NET
       M2213$layout<-layout_with_sugiyama(M2213,layers=V(M2213)$level)$layout

       M2221<-matrix(c("1","2","2","3","4","3","3","4","5","6"),5,2)
       M2221<-graph_from_edgelist(M2221)
       NET<-as.undirected(M2221)
       V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("5","6")]<-2
       M2221<-NET
       M2221$layout<-layout_with_sugiyama(M2221,layers=V(M2221)$level)$layout

       M2222<-matrix(c("1","1","2","2","3","4","3","4","3","4","5","6"),6,2)
       M2222<-graph_from_edgelist(M2222)
       NET<-as.undirected(M2222)
       V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("5","6")]<-2
       M2222<-NET
       M2222$layout<-layout_with_sugiyama(M2222,layers=V(M2222)$level)$layout

       M2223<-matrix(c("1","2","3","3","4","3","4","5","6","6"),5,2)
       M2223<-graph_from_edgelist(M2223)
       NET<-as.undirected(M2223)
       V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("5","6")]<-2
       M2223<-NET
       M2223$layout<-layout_with_sugiyama(M2223,layers=V(M2223)$level)$layout

       M2224<-matrix(c("1","2","2","3","3","4","3","3","4","5","6","6"),6,2)
       M2224<-graph_from_edgelist(M2224)
       NET<-as.undirected(M2224)
       V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("5","6")]<-2
       M2224<-NET
       M2224$layout<-layout_with_sugiyama(M2224,layers=V(M2224)$level)$layout

       M2225<-matrix(c("1","1","2","2","3","4","4","3","4","3","4","5","5","6"),7,2)
       M2225<-graph_from_edgelist(M2225)
       NET<-as.undirected(M2225)
       V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("5","6")]<-2
       M2225<-NET
       M2225$layout<-layout_with_sugiyama(M2225,layers=V(M2225)$level)$layout

       M2226<-matrix(c("1","2","3","3","4","4","3","4","5","6","5","6"),6,2)
       M2226<-graph_from_edgelist(M2226)
       NET<-as.undirected(M2226)
       V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("5","6")]<-2
       M2226<-NET
       M2226$layout<-layout_with_sugiyama(M2226,layers=V(M2226)$level)$layout

       M2227<-matrix(c("1","2","2","3","3","4","4","3","3","4","5","6","5","6"),7,2)
       M2227<-graph_from_edgelist(M2227)
       NET<-as.undirected(M2227)
       V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("5","6")]<-2
       M2227<-NET
       M2227$layout<-layout_with_sugiyama(M2227,layers=V(M2227)$level)$layout

       M2228<-matrix(c("1","1","2","2","3","3","4","4","3","4","3","4","5","6","5","6"),8,2)
       M2228<-graph_from_edgelist(M2228)
       NET<-as.undirected(M2228)
       V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("5","6")]<-2
       M2228<-NET
       M2228$layout<-layout_with_sugiyama(M2228,layers=V(M2228)$level)$layout

       M2229<-matrix(c("1","2","2","3","4","4","3","3","4","5","5","6"),6,2)
       M2229<-graph_from_edgelist(M2229)
       NET<-as.undirected(M2229)
       V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("5","6")]<-2
       M2229<-NET
       M2229$layout<-layout_with_sugiyama(M2229,layers=V(M2229)$level)$layout

       M3211<-matrix(c("1","2","3","4","5","4","4","5","6","6"),5,2)
       M3211<-graph_from_edgelist(M3211)
       NET<-as.undirected(M3211)
       V(NET)$level[(V(NET)$name)%in%c("1","2","3")]<-0
       V(NET)$level[(V(NET)$name)%in%c("5","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("6")]<-2
       M3211<-NET
       M3211$layout<-layout_with_sugiyama(M3211,layers=V(M3211)$level)$layout

       M3212<-matrix(c("1","1","2","3","4","5","4","5","5","5","6","6"),6,2)
       M3212<-graph_from_edgelist(M3212)
       NET<-as.undirected(M3212)
       V(NET)$level[(V(NET)$name)%in%c("1","2","3")]<-0
       V(NET)$level[(V(NET)$name)%in%c("5","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("6")]<-2
       M3212<-NET
       M3212$layout<-layout_with_sugiyama(M3212,layers=V(M3212)$level)$layout

       M3213<-matrix(c("1","2","2","3","4","5","4","4","5","5","6","6"),6,2)
       M3213<-graph_from_edgelist(M3213)
       NET<-as.undirected(M3213)
       V(NET)$level[(V(NET)$name)%in%c("1","2","3")]<-0
       V(NET)$level[(V(NET)$name)%in%c("5","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("6")]<-2
       M3213<-NET
       M3213$layout<-layout_with_sugiyama(M3213,layers=V(M3213)$level)$layout

       M3214<-matrix(c("1","1","2","2","3","4","5","4","5","4","5","5","6","6"),7,2)
       M3214<-graph_from_edgelist(M3214)
       NET<-as.undirected(M3214)
       V(NET)$level[(V(NET)$name)%in%c("1","2","3")]<-0
       V(NET)$level[(V(NET)$name)%in%c("5","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("6")]<-2
       M3214<-NET
       M3214$layout<-layout_with_sugiyama(M3214,layers=V(M3214)$level)$layout

       M3215<-matrix(c("1","1","2","2","3","3","4","5","4","5","4","5","4","5","6","6"),8,2)
       M3215<-graph_from_edgelist(M3215)
       NET<-as.undirected(M3215)
       V(NET)$level[(V(NET)$name)%in%c("1","2","3")]<-0
       V(NET)$level[(V(NET)$name)%in%c("5","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("6")]<-2
       M3215<-NET
       M3215$layout<-layout_with_sugiyama(M3215,layers=V(M3215)$level)$layout

       M131<-matrix(c("1","1","1","2","3","4","2","3","4","5","5","5"),6,2)
       M131<-graph_from_edgelist(M131)
       NET<-as.undirected(M131)
       V(NET)$level[(V(NET)$name)%in%c("1")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3","2","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("5")]<-2
       M131<-NET
       M131$layout<-layout_with_sugiyama(M131,layers=V(M131)$level)$layout

       M1321<-matrix(c("1","1","1","2","3","4","2","3","4","5","5","6"),6,2)
       M1321<-graph_from_edgelist(M1321)
       NET<-as.undirected(M1321)
       V(NET)$level[(V(NET)$name)%in%c("1")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3","2","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("6","5")]<-2
       M1321<-NET
       M1321$layout<-layout_with_sugiyama(M1321,layers=V(M1321)$level)$layout

       M1322<-matrix(c("1","1","1","2","2","3","4","2","3","4","5","6","6","6"),7,2)
       M1322<-graph_from_edgelist(M1322)
       NET<-as.undirected(M1322)
       V(NET)$level[(V(NET)$name)%in%c("1")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3","2","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("6","5")]<-2
       M1322<-NET
       M1322$layout<-layout_with_sugiyama(M1322,layers=V(M1322)$level)$layout

       M1323<-matrix(c("1","1","1","2","3","3","4","2","3","4","5","5","6","6"),7,2)
       M1323<-graph_from_edgelist(M1323)
       NET<-as.undirected(M1323)
       V(NET)$level[(V(NET)$name)%in%c("1")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3","2","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("6","5")]<-2
       M1323<-NET
       M1323$layout<-layout_with_sugiyama(M1323,layers=V(M1323)$level)$layout

       M1324<-matrix(c("1","1","1","2","2","3","3","4","2","3","4","5","6","5","6","6"),8,2)
       M1324<-graph_from_edgelist(M1324)
       NET<-as.undirected(M1324)
       V(NET)$level[(V(NET)$name)%in%c("1")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3","2","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("6","5")]<-2
       M1324<-NET
       M1324$layout<-layout_with_sugiyama(M1324,layers=V(M1324)$level)$layout

       M1325<-matrix(c("1","1","1","2","2","3","3","4","4","2","3","4","5","6","5","6","5","6"),9,2)
       M1325<-graph_from_edgelist(M1325)
       NET<-as.undirected(M1325)
       V(NET)$level[(V(NET)$name)%in%c("1")]<-0
       V(NET)$level[(V(NET)$name)%in%c("3","2","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("6","5")]<-2
       M1325<-NET
       M1325$layout<-layout_with_sugiyama(M1325,layers=V(M1325)$level)$layout

       M2311<-matrix(c("1","1","2","3","4","5","3","4","5","6","6","6"),6,2)
       M2311<-graph_from_edgelist(M2311)
       NET<-as.undirected(M2311)
       V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
       V(NET)$level[(V(NET)$name)%in%c("5","3","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("6")]<-2
       M2311<-NET
       M2311$layout<-layout_with_sugiyama(M2311,layers=V(M2311)$level)$layout

       M2312<-matrix(c("1","2","2","2","3","4","5","3","3","4","5","6","6","6"),7,2)
       M2312<-graph_from_edgelist(M2312)
       NET<-as.undirected(M2312)
       V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
       V(NET)$level[(V(NET)$name)%in%c("5","3","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("6")]<-2
       M2312<-NET
       M2312$layout<-layout_with_sugiyama(M2312,layers=V(M2312)$level)$layout

       M2313<-matrix(c("1","1","2","2","3","4","5","3","4","4","5","6","6","6"),7,2)
       M2313<-graph_from_edgelist(M2313)
       NET<-as.undirected(M2313)
       V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
       V(NET)$level[(V(NET)$name)%in%c("5","3","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("6")]<-2
       M2313<-NET
       M2313$layout<-layout_with_sugiyama(M2313,layers=V(M2313)$level)$layout

       M2314<-matrix(c("1","1","1","2","2","3","4","5","3","4","5","4","5","6","6","6"),8,2)
       M2314<-graph_from_edgelist(M2314)
       NET<-as.undirected(M2314)
       V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
       V(NET)$level[(V(NET)$name)%in%c("5","3","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("6")]<-2
       M2314<-NET
       M2314$layout<-layout_with_sugiyama(M2314,layers=V(M2314)$level)$layout

       M2315<-matrix(c("1","1","1","2","2","2","3","4","5","3","4","5","3","4","5","6","6","6"),9,2)
       M2315<-graph_from_edgelist(M2315)
       NET<-as.undirected(M2315)
       V(NET)$level[(V(NET)$name)%in%c("1","2")]<-0
       V(NET)$level[(V(NET)$name)%in%c("5","3","4")]<-1
       V(NET)$level[(V(NET)$name)%in%c("6")]<-2
       M2315<-NET
       M2315$layout<-layout_with_sugiyama(M2315,layers=V(M2315)$level)$layout

       M141<-matrix(c("1","1","1","1","2","3","4","5","2","3","4","5","6","6","6","6"),8,2)
       M141<-graph_from_edgelist(M141)
       NET<-as.undirected(M141)
       V(NET)$level[(V(NET)$name)%in%c("1")]<-0
       V(NET)$level[(V(NET)$name)%in%c("2","3","4","5")]<-1
       V(NET)$level[(V(NET)$name)%in%c("6")]<-2
       M141<-NET
       M141$layout<-layout_with_sugiyama(M141,layers=V(M141)$level)$layout

       return(list(M111,M112,M113,M114,M211,M212,M213,M311,M312,M411,M1211,M1221,M1222,M1223,M1231,M1232,M1233,M1234,
                   M1235,M2211,M2212,M2213,M2221,M2222,M2223,M2224,M2225,M2226,M2227,M2228,M2229,M3211,
                   M3212,M3213,M3214,M3215,M131,M1321,M1322,M1323,M1324,M1325,M2311,M2312,M2313,M2314,M2315,M141))
    }
}




