% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/GRAB_Simu.R
\name{GRAB.makePlink}
\alias{GRAB.makePlink}
\title{Make PLINK files using a numeric R matrix}
\usage{
GRAB.makePlink(
  GenoMat,
  OutputPrefix,
  A1 = "G",
  A2 = "A",
  CHR = NULL,
  BP = NULL,
  Pheno = NULL,
  Sex = NULL
)
}
\arguments{
\item{GenoMat}{a numeric \code{n*m} genotype matrix (0,1,2,-9). Each row is for one subject and each column is for one marker. Row names of subject IDs and column names of marker IDs are required.}

\item{OutputPrefix}{a character, prefix of the PLINK files to output (including path).}

\item{A1}{a character to specify allele 1 (\emph{default="G"}), usually minor (ALT).}

\item{A2}{a character to specify allele 2 (\emph{default="A"}), usually major (REF).}

\item{CHR}{a character vector of the chromosome numbers for all markers. \emph{Default=NULL}, that is, \code{CHR=rep(1, m)}.}

\item{BP}{a numeric vector of the base positions for all markers. \emph{Default=NULL}, that is, \code{BP=1:m)}.}

\item{Pheno}{a character vector of the phenotypes for all subjects. \emph{Default=NULL}, that is, \code{Pheno=rep(-9, n)}.}

\item{Sex}{a numeric vector of the sex for all subjects. \emph{Default=NULL}, that is, \code{Sex=rep(1, n))}.}
}
\value{
\code{PLINK} text files (PED and MAP) are stored in 'OutputPrefix'. Suppose A1 is "G" and A2 is "A", then genotype of 0,1,2,-9 will be coded as "GG", "AG", "AA", "00". If PLINK binary files (BED, BIM, and FAM) are required, please download PLINK software and use option of "--make-bed".
Please check \code{Details} section for the downstream process.
}
\description{
Make PLINK files using a numeric matrix \code{GenoMat} (0,1,2,-9), \code{rownames(GenoMat)} are subject IDs and \code{colnames(GenoMat)} are marker IDs
}
\details{
Check \href{https://www.cog-genomics.org/plink/2.0/}{link} for detailed information of \code{PLINK} 2.00 alpha. Check \href{https://enkre.net/cgi-bin/code/bgen/doc/trunk/doc/wiki/bgenix.md}{link} for detailed information of \code{bgenix} tool.
\subsection{Convert PLINK text files to binary files}{

Run \code{plink --file simuPLINK --make-bed --out simuPLINK} to convert PLINK text files (MAP and PED) to binary files (BED, BIM, and FAM).
}

\subsection{Convert PLINK binary files to raw files}{

Run \code{plink --bfile simuPLINK --recode A --out simuRAW} to convert PLINK binary files (BED, BIM, and FAM) to raw files (raw).
}

\subsection{Convert PLINK binary files to bgen files}{

RUN \code{plink2 --bfile simuPLINK --export bgen-1.2 bits=8 ref-first --out simuBGEN} to convert PLINK binary files (BED, BIM, and FAM) to BGEN binary files (BGEN).
}

\subsection{Make bgi file using \code{bgenix} tool}{

RUN \code{bgenix -g simuBGEN.bgen --index}
}
}
\examples{
### Step 1: simulate a numeric genotype matrix
n <- 1000
m <- 20
MAF <- 0.3
set.seed(123)
GenoMat <- matrix(rbinom(n * m, 2, MAF), n, m)
rownames(GenoMat) <- paste0("Subj-", 1:n)
colnames(GenoMat) <- paste0("SNP-", 1:m)
OutputDir <- tempdir()
outputPrefix <- file.path(OutputDir, "simuPLINK")

### Step 2(a): make PLINK files without missing genotype
GRAB.makePlink(GenoMat, outputPrefix)

### Step 2(b): make PLINK files with genotype missing rate of 0.1
indexMissing <- sample(n * m, 0.1 * n * m)
GenoMat[indexMissing] <- -9
GRAB.makePlink(GenoMat, outputPrefix)

## The following are in shell environment
# plink --file simuPLINK --make-bed --out simuPLINK
# plink --bfile simuPLINK --recode A --out simuRAW
# plink2 --bfile simuPLINK --export bgen-1.2 bits=8 ref-first --out simuBGEN
# UK Biobank use 'ref-first'
# bgenix -g simuBGEN.bgen --index


}
