
# not scaled and centered
sample_finite_fac <- \(mu, func.L, costm, n) {
    nu <- emp_measure(n, mu)
    Lnu <- func.L(nu)
    ot_cost_sgn_rowwise_posneg(Lnu, costm)
}

sample_limit_fac_null_plugin <- \(func.L, costm, gen_G) {
    LG <- gen_G() |> func.L()
    G_u <- pos(LG)
    G_v <- neg(LG)
    future_sapply(seq_len(nrow(LG)), \(m) had_deriv_ot_cost_sgn_null(G_u[m, ], G_v[m, ], costm))
}

sample_limit_fac_null_boot_m <- \(mu, func.L, costm, m) {
    sample_finite_fac(mu, func.L, costm, m)
}

sample_limit_fac_null_boot_deriv <- \(mu, func.L, costm, gen_G_boot) {
    sample_limit_fac_null_plugin(func.L, costm, gen_G_boot)
}

sample_limit_fac_alt <- \(Lmu.pos, Lmu.neg, func.L, costm, gen_G) {
    LG <- gen_G() |> func.L()
    G_u <- pos(LG)
    G_v <- neg(LG)
    future_sapply(seq_len(nrow(LG)), \(m) had_deriv_ot_cost_alt(Lmu.pos[m, ], Lmu.neg[m, ], G_u[m, ], G_v[m, ], costm))
}

##

#' @rdname sim_fac
#' @order 1
#' @export
simulate_finite_FDOTT <- \(mu, costm, n, H0 = "*", num.sim = 1000) {
    check_numsim(num.sim)
    mu <- get_fac_mat(mu)
    check_mu(mu)
    K <- nrow(mu)
    n <- unlist(n)
    check_n(n, K)
    fac.lvls <- get_fac_lvls(mu)
    L <- get_Lmat(H0, fac.lvls)
    func.L <- \(x) L %*% x
    Lmu <- func.L(mu)
    s <- get_s(L)
    objval <- sum(ot_cost_sgn_rowwise_posneg(Lmu, costm) / s)

    rho <- limit_coeffs(n)$rho

    p <- progressr::progressor(steps = num.sim)
    ls <- future_replicate(
        num.sim,
        { x <- sum(sample_finite_fac(mu, func.L, costm, n) / s); p(); x }
    )
    sqrt(rho) * (ls - objval)
}

#' @rdname sim_fac
#' @order 3
#' @export
simulate_limit_FDOTT_alt <- \(mu, costm, delta, H0 = "*", num.sim = 1000) {
    check_numsim(num.sim)
    mu <- get_fac_mat(mu)
    check_mu(mu)
    K <- nrow(mu)
    N <- ncol(mu)
    check_cost_mat(costm, N)
    delta <- unlist(delta)
    check_delta(delta, K)
    gen_G <- get_gen_G(mu, delta, N, K)
    fac.lvls <- get_fac_lvls(mu)
    L <- get_Lmat(H0, fac.lvls)
    func.L <- \(x) L %*% x
    Lmu <- func.L(mu)
    Lmu.pos <- pos(Lmu)
    Lmu.neg <- neg(Lmu)
    s <- get_s(L)

    p <- progressr::progressor(steps = num.sim)
    future_replicate(
        num.sim,
        { x <- sum(sample_limit_fac_alt(Lmu.pos, Lmu.neg, func.L, costm, gen_G) / s); p(); x}
    )
}

# n is not used here, but needed such that simulate_limit_LR_null_*
# all have the same parameters in the same order
simulate_limit_FDOTT_null_plugin <- \(mu, func.L, costm, n, delta, num.sim, get.gen_G) {

    K <- nrow(mu)
    N <- ncol(mu)

    gen_G <- get.gen_G(mu, delta, N, K)

    p <- progressr::progressor(steps = num.sim)
    future_replicate(
        num.sim,
        { x <- sample_limit_fac_null_plugin(func.L, costm, gen_G); p(); x }
    )
}

simulate_limit_FDOTT_null_boot_m <- \(mu, func.L, costm, n, delta, num.sim, get.gen_G) {
    rho <- limit_coeffs(n)$rho

    p <- progressr::progressor(steps = num.sim)
    future_replicate(
        num.sim,
        { x <- sample_limit_fac_null_boot_m(mu, func.L, costm, n); p(); x }
    ) * sqrt(rho)
}

simulate_limit_FDOTT_null_boot_deriv <- \(mu, func.L, costm, n, delta, num.sim, get.gen_G) {
    rho <- limit_coeffs(n)$rho
    gen_G <- get_gen_G_boot(mu, rho, n)

    p <- progressr::progressor(steps = num.sim)
    future_replicate(
        num.sim,
        { x <- sample_limit_fac_null_boot_deriv(mu, func.L, costm, gen_G); p(); x }
    )
}

simulate_limit_FDOTT_null_one_way_perm <- \(samples, func.L, costm, n, rho, num.sim) {

    permute <- get_permute_emp_mat(samples, n)

    p <- progressr::progressor(steps = num.sim)
    sample_limit <- \() {
        nu <- permute() |> func.L()
        x <- ot_cost_sgn_rowwise_posneg(nu, costm)
        p()
        x
    }

    res <- future_replicate(
        num.sim,
        sample_limit()
    ) * sqrt(rho)

    if (is.null(dim(res))) {
        res <- matrix(res, 1, num.sim)
    }

    res
}

simulate_limit_FDOTT_null_base <- \(mu, func.L, costm, n, delta, num.sim, m.p, get.gen_G,
                                    method = c("plug-in", "bootstrap-deriv", "bootstrap-m")) {
    check_numsim(num.sim)
    check_mu(mu)
    K <- nrow(mu)
    N <- ncol(mu)
    check_cost_mat(costm, N)
    method <- match.arg(method)

    simulate_limit <- switch(
        method,
        "plug-in"         = simulate_limit_FDOTT_null_plugin,
        "bootstrap-m"     = simulate_limit_FDOTT_null_boot_m,
        "bootstrap-deriv" = simulate_limit_FDOTT_null_boot_deriv
    )

    if (method == "bootstrap-m") {
        check_n(n, K)
        stopifnot(is_num_scalar(m.p), 0 < m.p, m.p < 1)
        func.m <- \(n) floor(n^m.p)
        if (is.list(n)) {
            m <- rapply(n, func.m, how = "replace")
        } else {
            m <- sapply(n, func.m)
        }
    } else {
        check_delta(delta, K)
        m <- n
    }

    res <- simulate_limit(mu, func.L, costm, m, delta, num.sim, get.gen_G)

    if (is.null(dim(res))) {
        res <- matrix(res, 1, num.sim)
    }

    res
}

#' @title Simulations for `FDOTT`
#' @description Perform simulations for the test statistic used in [`FDOTT`].
#' @param mu matrix (row-wise) or nested list containing \eqn{K} probability vectors.
#' @param H0 null hypothesis, see [`FDOTT`] for more information.
#' @param costm semi-metric cost matrix \eqn{c \in \mathbb{R}^{N \times N}}.
#' @param n samples sizes. Must be of the same structure as `mu`.
#' @param delta asymptotic sample size ratios. Must be of the same structure as `mu`.
#' @param num.sim number of samples to draw from the limiting null distribution.
#' @param m.p exponent \eqn{p \in (0, 1)} used for `method = "bootstrap-m"`.
#' @param method the method to use to simulate from the null distribution.
#' @param mean mean of the Gaussians appearing in the limiting distribution. Must be of the same structure as `mu`.
#' @details See [`FDOTT`] for the definition of the test statistic and more details.
#'
#' `simulate_finite_FDOTT` simulates from the finite sample distribution.
#'
#' `simulate_limit_FDOTT_null` and `simulate_limit_FDOTT_alt` simulate from the limiting distribution under the null or alternative, respectively.
#'
#' All these simulations can be done in parallel via [`future::plan`] and the progress can be shown with [`progressr::with_progress`].
#' @returns A vector containing the simulated samples.
#' @rdname sim_fac
#' @order 2
#' @example examples/sim_fd.R
#' @export
simulate_limit_FDOTT_null <- \(mu, costm, n = NULL, delta = NULL, H0 = "*", num.sim = 1000,
                               method = c("plug-in", "bootstrap-deriv", "bootstrap-m"), m.p = 0.5, mean = NULL) {
    method <- match.arg(method)

    mu <- get_fac_mat(mu)
    delta <- unlist(delta)
    n <- unlist(n)
    fac.lvls <- get_fac_lvls(mu)
    L <- get_Lmat(H0, fac.lvls)
    func.L <- \(x) L %*% x
    s <- get_s(L)

    if (!is.null(mean)) {
        mean <- get_fac_mat(mean)
    }

    ls <- simulate_limit_FDOTT_null_base(mu, func.L, costm, n, delta, num.sim, m.p,
                                         add_mean_to_gen(get_gen_G, mean), method = method)

    sweep(ls, 1, s, "/") |> apply(2, sum)
}
