\name{estPOT}
\alias{estPOT}
\title{Estimation of generalized Pareto distributions}
\usage{
estPOT(
  data,
  k = 10L,
  pn = NULL,
  type = c("continuous", "discrete"),
  method = c("bayesian", "frequentist"),
  prior = "empirical",
  start = NULL,
  sig0 = NULL,
  nsim = 5000L,
  burn = 1000L,
  p = 0.234,
  optim.method = "Nelder-Mead",
  control = NULL,
  ...
)
}
\arguments{
\item{data}{numeric vector of length n containing complete data values (under and above threshold)}

\item{k}{double indicating the effective sample size. Default: 10}

\item{pn}{numeric vector containing one or more percentile level at which extreme quantiles are estimated, with \eqn{p < k/n}. Default: NULL}

\item{type}{string indicating distribution types. Default: \code{c('continuous','discrete')}}

\item{method}{string indicating estimation methods. Default: \code{c('bayesian', 'frequentist')}}

\item{prior}{string indicating prior distribution (uniform or empirical). Default: \code{'empirical'}}

\item{start}{list of 2 containing starting values for scale and shape parameters. Default: \code{NULL}}

\item{sig0}{double indicating the initial value for the update of the variance in the MCMC algorithm. Default: \code{NULL}}

\item{nsim}{double indicating the total number of iterations of the MCMC algorithm in the Bayesian estimation case. Default: 5000L}

\item{burn}{double indicating the number of iterations to exclude in the MCMC algorithm of the Bayesian estimation case. Default: 1000L}

\item{p}{double indicating the desired overall sampler acceptance probability. Default: 0.234}

\item{optim.method}{string indicating the optimization method in the frequentist estimation case. Default: \code{'Nelder-Mead'}}

\item{control}{list containing additional parameters for the minimization function \link[stats]{optim}. Default: \code{NULL}}

\item{...}{other arguments passed to the function}
}
\value{
a list with the following elements
\itemize{
\item Bayesian estimation case
\itemize{
\item \code{Q.est} matrix with \code{nsim-burn} rows and \code{length(pn)} columns containing the posterior sample of the
extreme quantile estimated at level given in pn
\item \code{post_sample} matrix with \code{nsim-burn} rows and 2 columns containing the posterior sample of the scale and
shape parameters for the continuous or discrete generalized Pareto distribution
\item \code{burn} double indicating the number of iterations excluded in the MCMC algorithm
\item \code{straight.reject} vector of length \code{nsim-burn+1} indicating the iterations in which the proposed parameters do not respect basic constraints
\item \code{sig.vec} vector of length \code{nsim}\eqn{-\lfloor(5 / (p (1 - p)))\rfloor+1} containing the values of the variance updated at each iteration of the MCMC algorithm
\item \code{accept.prob} matrix containing the values of acceptance probability (second column) corresponding to specific iterations (first column)
\item \code{msg} character string containing an output message on the result of the Bayesian estimation procedure
\item \code{mle} vector of length 2 containing the maximum likelihood estimates of the scale and shape parameters of the continuous or discrete generalized Pareto distribution
\item \code{t} double indicating the threshold for the generalized Pareto model, corresponding to the \eqn{n-k}th order statistic of the sample
}
\item Frequentist estimation case
\itemize{
\item \code{est} vector of length 2 containing the maximum likelihood estimates of the scale and shape parameters of the continuous or discrete generalized Pareto distribution
\item \code{t} double indicating the threshold
\item \code{Q.est} vector of dimension \code{length(pn)} containing the estimates of the extreme quantile at level given in pn
\item \code{VarCov} \eqn{2 \times 2} variance-covariance matrix of (gamma, sigma)
\item \code{Q.VC} variance of Q.est
\item \code{msg} character string containing an output message on the result of the frequentist estimation procedure
}
}
}
\description{
Bayesian or frequentist estimation of the scale and shape parameters of the continuous or discrete generalized Pareto distribution.
}
\examples{
\dontrun{
# generate data
set.seed(1234)
n <- 500
samp <- evd::rfrechet(n,0,3,4)
# set effective sample size and threshold
k <- 50
threshold <- sort(samp,decreasing = TRUE)[k+1]
# preliminary mle estimates of scale and shape parameters
mlest <- evd::fpot(samp, threshold)
# empirical bayes procedure
proc <- estPOT(
  samp,
  k = k,
  pn = c(0.01, 0.005),
  type = "continuous",
  method = "bayesian",
  prior = "empirical",
  start = as.list(mlest$estimate),
  sig0 = 0.1)
 }
}
\references{
Dombry, C., S. Padoan and S. Rizzelli (2025). Asymptotic theory for Bayesian inference and prediction: from the ordinary to a conditional Peaks-Over-Threshold method, arXiv:2310.06720v2.
}
\seealso{
\code{\link[evd]{fpot}}
}
