% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/HYPERPO.R
\name{HYPERPO}
\alias{HYPERPO}
\title{The hyper Poisson family}
\usage{
HYPERPO(mu.link = "log", sigma.link = "log")
}
\arguments{
\item{mu.link}{defines the mu.link, with "log" link as the default for the mu parameter.}

\item{sigma.link}{defines the sigma.link, with "log" link as the default for the sigma.}
}
\value{
Returns a \code{gamlss.family} object which can be used
to fit a hyper-Poisson distribution
in the \code{gamlss()} function.
}
\description{
The function \code{HYPERPO()} defines the
hyper Poisson distribution
a two parameter distribution,
for a \code{gamlss.family} object to be used in GAMLSS
fitting using the function \code{gamlss()}.
}
\details{
The hyper-Poisson distribution with parameters \eqn{\mu} and \eqn{\sigma}
has a support 0, 1, 2, ... and density given by

\eqn{f(x | \mu, \sigma) = \frac{\mu^x}{_1F_1(1;\mu;\sigma)}\frac{\Gamma(\sigma)}{\Gamma(x+\sigma)}}

where the function \eqn{_1F_1(a;c;z)} is defined as

\eqn{_1F_1(a;c;z) = \sum_{r=0}^{\infty}\frac{(a)_r}{(c)_r}\frac{z^r}{r!}}

and \eqn{(a)_r = \frac{\gamma(a+r)}{\gamma(a)}} for \eqn{a>0} and \eqn{r} positive integer.

Note: in this implementation we changed the original parameters \eqn{\lambda} and \eqn{\gamma}
for \eqn{\mu} and \eqn{\sigma} respectively, we did it to implement this distribution within gamlss framework.
}
\examples{
# Example 1
# Generating some random values with
# known mu and sigma
set.seed(1234)
y <- rHYPERPO(n=200, mu=10, sigma=1.5)

# Fitting the model
library(gamlss)
mod1 <- gamlss(y~1, sigma.fo=~1, family=HYPERPO,
               control=gamlss.control(n.cyc=500, trace=TRUE))

# Extracting the fitted values for mu and sigma
# using the inverse link function
exp(coef(mod1, what="mu"))
exp(coef(mod1, what="sigma"))

# Example 2
# Generating random values under some model

# A function to simulate a data set with Y ~ HYPERPO
gendat <- function(n) {
  x1 <- runif(n)
  x2 <- runif(n)
  mu    <- exp(1.21 - 3 * x1) # 0.75 approximately
  sigma <- exp(1.26 - 2 * x2) # 1.30 approximately
  y <- rHYPERPO(n=n, mu=mu, sigma=sigma)
  data.frame(y=y, x1=x1, x2=x2)
}

set.seed(1234)
dat <- gendat(n=100)

mod2 <- gamlss(y~x1, sigma.fo=~x2, family=HYPERPO, data=dat,
                 control=gamlss.control(n.cyc=500, trace=TRUE))

summary(mod2)
}
\references{
Sáez-Castillo, A. J., & Conde-Sánchez, A. (2013). A hyper-Poisson regression
model for overdispersed and underdispersed count data.
Computational Statistics & Data Analysis, 61, 148-157.
}
\seealso{
\link{dHYPERPO}.
}
\author{
Freddy Hernandez, \email{fhernanb@unal.edu.co}
}
