% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kernel_regression_estimator.R
\name{truncated_est}
\alias{truncated_est}
\title{Compute the Truncated Kernel Regression Estimator.}
\usage{
truncated_est(
  X,
  x,
  t,
  T1,
  T2,
  b,
  kernel_name = "gaussian",
  kernel_params = c(),
  pd = TRUE,
  type = "autocovariance",
  meanX = mean(X),
  custom_kernel = FALSE
)
}
\arguments{
\item{X}{A vector representing observed values of the time series.}

\item{x}{A vector of lags.}

\item{t}{The arguments at which the autocovariance function is calculated at.}

\item{T1}{The first truncation point, \eqn{T_{1} > 0.}}

\item{T2}{The second truncation point, \eqn{T_{2} > T_{1} > 0.}}

\item{b}{Bandwidth parameter, greater than 0.}

\item{kernel_name}{The name of the symmetric kernel (see \link{kernel_symm}) function to be used. Possible values are:
gaussian, wave, rational_quadratic, and bessel_j. Alternatively, a custom kernel function can be provided, see the examples.}

\item{kernel_params}{A vector of parameters of the kernel function. See \link{kernel_symm} for parameters.}

\item{pd}{Whether a positive-definite estimate should be used. Defaults to \code{TRUE}.}

\item{type}{Compute either the 'autocovariance' or 'autocorrelation'. Defaults to 'autocovariance'.}

\item{meanX}{The average value of \code{X}. Defaults to \code{mean(X)}.}

\item{custom_kernel}{If a custom kernel is to be used or not. Defaults to \code{FALSE}.}
}
\value{
A vector whose values are the truncated kernel regression estimates.
}
\description{
This function computes the truncated kernel regression estimator, based on the kernel regression estimator \eqn{\hat{\rho}(\cdot)}, see \link{adjusted_est}.
}
\details{
This function computes the truncated kernel regression estimator,
\deqn{\hat{\rho}_{1}(t) = \left\{ \begin{array}{ll}
\hat{\rho}(t) & 0 \leq t \leq T_{1} \\
\hat{\rho}(T_{1}) (T_{2} - t)(T_{2} - T_{1})^{-1} & T_{1} < t \leq T_{2} \\
0 & t > T_{2}
\end{array} \right. }
where \eqn{\hat{\rho}(\cdot)} is the kernel regression estimator, see \link{adjusted_est}.

Compared to \link{adjusted_est}, this function brings down the estimate to zero linearly between \eqn{T_{1}} and \eqn{T_{2}}.
In the case of short-range dependence, this may be beneficial as it can remove estimation artefacts at large lags.

To make this estimator positive-definite, the following procedure is used:
\enumerate{
\item Take the discrete Fourier cosine transform
\eqn{\widehat{\mathcal{F}}(\theta)}.
\item Find the smallest frequency where its associated value in the spectral domain is negative
\deqn{\hat{\theta} = \inf\{ \theta > 0 :  \widehat{\mathcal{F}}(\theta)) < 0\}.}
\item Set all values starting at the frequency to zero.
\item Perform the Fourier inversion.
}

If \eqn{\hat{\theta}} is a small frequency, most of the spectrum equals zero, resulting in an inaccurate estimate of the autocovariance function, see Bilchouris and Olenko (2025).
}
\examples{
X <- c(1, 2, 3, 4)
truncated_est(X, 1:4, 1:3, 1, 2, 0.1,
                  "gaussian")

my_kernel <- function(x, theta, params) {
  stopifnot(theta > 0, length(x) >= 1)
  return(exp(-((abs(x) / theta)^params[1])) * (2 * theta  * gamma(1 + 1/params[1])))
}
truncated_est(X, 1:4, 1:3, 1, 2, 0.1, my_kernel, c(0.25), custom_kernel = TRUE)
}
\references{
Hall, P. & Patil, P. (1994). Properties of nonparametric estimators of autocovariance for stationary random fields. Probability Theory and Related Fields 99(3), 399-424. https://doi.org/10.1007/bf01199899

Hall, P., Fisher, N. I., & Hoffmann, B. (1994). On the nonparametric estimation of covariance functions. The Annals of Statistics 22(4), 2115-2134. https://doi.org/10.1214/aos/1176325774

Bilchouris, A. & Olenko, A (2025). On Nonparametric Estimation of Covariogram. Austrian Statistical Society 54(1), 112–137. https://doi.org/10.17713/ajs.v54i1.1975
}
