#' Initialize input vectors for the CMTF algorithm
#'
#' @param Z List object as generated by [setupCMTFdata()].
#' @param numComponents Integer stating the number of desired components for the CMTF model.
#' @param initialization Initialization method, either "random" or "nvec" (default "random"). Random will initialize random input vectors. Nvec will initialize vectors according to an singular value decomposition of the (matricized, if needed) concatenated datasets per mode.
#' @param output How to return output: as a "Fac" object (default) or vectorized ("vect").
#' @param Y Used as dependent variable when initialization is set to "npls". Not used by default.
#'
#' @return List or vector of initialized input vectors per mode.
#' @export
#'
#' @examples
#' set.seed(123)
#' A = array(rnorm(108*2), c(108, 2))
#' B = array(rnorm(100*2), c(100, 2))
#' C = array(rnorm(10*2), c(10, 2))
#' D = array(rnorm(100*2), c(100,2))
#' E = array(rnorm(10*2), c(10,2))
#'
#' df1 = reinflateTensor(A, B, C)
#' df2 = reinflateTensor(A, D, E)
#' datasets = list(df1, df2)
#' modes = list(c(1,2,3), c(1,4,5))
#' Z = setupCMTFdata(datasets, modes, normalize=FALSE)
#'
#' init = initializeCMTF(Z, 1)
initializeCMTF = function(Z, numComponents, initialization="random", output="Fac", Y=NULL){

  init = list()
  numModes = max(unlist(Z$modes))
  numDatasets = length(Z$object)

  if(initialization == "random"){
    for(i in 1:numModes){
      size = Z$sizes[i]
      init[[i]] = array(stats::rnorm(size*numComponents), c(size, numComponents))
    }
  } else if(initialization == "nvec"){

    for(i in 1:numModes){
      eligibleDatasets = list()
      iterator = 1
      size = Z$sizes[i]

      for(p in 1:numDatasets){
        if(i %in% Z$modes[[p]]){
          eligibleDatasets[[iterator]] = rTensor::k_unfold(Z$object[[p]], which(Z$modes[[p]] == i))@data
          iterator = iterator + 1
        }
      }

      df = do.call(cbind, eligibleDatasets)
      init[[i]] = svd(df, numComponents)$u
    }
  } #else if(initialization == "npls"){
#
#     for(p in 1:numDatasets){
#       fit = sNPLS::sNPLS(Z$object[[p]]@data, Y, ncomp=numComponents, threshold_j=0, threshold_k=0, scale.X=FALSE, center.X=FALSE, silent=TRUE, method="sNPLS")
#       relevantModes = Z$modes[[p]]
#
#       for(i in 1:length(relevantModes)){
#         init[[relevantModes[i]]] = matrix(fit[[i]], dim(fit[[i]]))
#       }
#     }
#   }

  if(output=="vect"){
    return(fac_to_vect(init))
  }
  else{
    return(init)
  }
}
