% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BAT.R
\name{optim.beta}
\alias{optim.beta}
\title{Optimization of beta diversity sampling protocols.}
\usage{
optim.beta(
  comm,
  tree,
  methods,
  base,
  seq = FALSE,
  abund = TRUE,
  runs = 1000,
  prog = TRUE
)
}
\arguments{
\item{comm}{A samples x species x sites array, with either abundance or incidence data.}

\item{tree}{A phylo or hclust object (used only for PD or FD) or alternatively a species x traits matrix or data.frame to build a functional tree.}

\item{methods}{A data.frame with the method names (1st column), number of samples per method (2nd column), base cost per method (3rd column, those costs that are fixed once a method is decided), and sample cost per method (those costs that add with each sample of the method, 4th column). If the last two columns are not provided base = 0 and sample = 1. The order of methods must be the same as in comm and the sum of the samples must be the same as nrow(comm).}

\item{base}{Allows defining a base mandatory protocol from which to build upon (complementarity analysis). It should be a vector with length = number of methods.}

\item{seq}{By default all combinations will be tested. If TRUE, a sequential approach will be taken, where methods are added based on the previous step. The method added will be the one providing the highest efficiency as quantified by the slope of the accumulation curve.}

\item{abund}{A boolean (T/F) indicating whether abundance data should be used (TRUE) or converted to incidence (FALSE) before analysis.}

\item{runs}{Number of random permutations to be made to the sample order. Default is 1000.}

\item{prog}{Present a text progress bar in the R console.}
}
\value{
A matrix of samples x methods (values being optimum number of samples per method). The last column is precision = (1 - average absolute difference from real beta).
}
\description{
Optimization of beta diversity sampling protocols when different methods and multiple samples per method are available.
}
\details{
Often, comparing differences between sites or the same site along time (i.e. measure beta diversity) it is not necessary to sample exhaustively. A minimum combination of samples targeting different sub-communities (that may behave differently) may be enough to perceive such differences, for example, for monitoring purposes.
Cardoso et al. (2024) introduce and differentiate the concepts of alpha-sampling and beta-sampling. While alpha-sampling optimization implies maximizing local diversity sampled (Cardoso 2009), beta-sampling optimization implies minimizing differences in beta diversity values between partially and completely sampled communities.
This function uses as beta diversity measures the Btotal, Brepl, Brich, Bgain and Bloss partitioning frameworks (Carvalho et al. 2012; Legendre 2019) and respective generalizations to PD and FD (Cardoso et al. 2014).
PD and FD are calculated based on a tree (hclust or phylo object, no need to be ultrametric).
}
\examples{
comm1 <- matrix(c(1,1,0,2,4,0,0,1,2,0,0,3), nrow = 4, ncol = 3, byrow = TRUE)
comm2 <- matrix(c(2,2,0,3,1,0,0,0,5,0,0,2), nrow = 4, ncol = 3, byrow = TRUE)
comm3 <- matrix(c(2,0,0,3,1,0,0,0,5,0,0,2), nrow = 4, ncol = 3, byrow = TRUE)
comm <- array(c(comm1, comm2, comm3), c(4,3,3))
colnames(comm) <- c("sp1","sp2","sp3")

methods <- data.frame(method = c("Met1","Met2","Met3"),
           nSamples = c(1,2,1), fixcost = c(1,1,2), varCost = c(1,1,1))
tree <- tree.build(gower(1:3))
tree$tip.label <- colnames(comm)

\dontrun{
  optim.beta(comm,,methods)
  optim.beta(comm,,methods, seq = TRUE)
  optim.beta(comm, tree, methods)
  optim.beta(comm,, methods = methods, seq = TRUE, base = c(0,1,1))
}
}
\references{
Cardoso, P. (2009) Standardization and optimization of arthropod inventories - the case of Iberian spiders. Biodiversity and Conservation, 18, 3949-3962.

Cardoso, P., Rigal, F., Carvalho, J.C., Fortelius, M., Borges, P.A.V., Podani, J. & Schmera, D. (2014) Partitioning taxon, phylogenetic and functional beta diversity into replacement and richness difference components. Journal of Biogeography, 41, 749-761.

Cardoso et al. (2024) Optimal inventorying and monitoring of taxonomic, phylogenetic, and functional diversity. PLoS One, 19: 0307156.

Carvalho, J.C., Cardoso, P. & Gomes, P. (2012) Determining the relative roles of species replacement and species richness differences in generating beta-diversity patterns. Global Ecology and Biogeography, 21, 760-771.
}
