% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ANOPA-anopa.R
\name{anopafct}
\alias{anopafct}
\alias{anopa}
\title{ANOPA: analysis of proportions using Anscombe transform.}
\usage{
anopa(formula = NULL, data = NULL, WSFactors = NULL)
}
\arguments{
\item{formula}{A formula with the factors on the left-hand side. See below for writing the
formula to match the data format.}

\item{data}{Dataframe in one of wide, long, or compiled format;}

\item{WSFactors}{For within-subject designs, provide the factor names and their number of levels.
This is expressed as a vector of strings such as "Moment(2)".}
}
\value{
An omnibus analyses of the given proportions. Each factor's significance is
assessed, as well as their interactions when there is more than one factor.
The results are obtained with \code{summary()} or \code{summarize()} as usual. If desired,
the corrected-only statistics can be presented \insertCite{w76}{ANOPA} using
\code{corrected()}; the uncorrected statistics only are obtained with \code{uncorrected()}.
For decomposition of the main analyses, follow the main analysis with \code{emProportions()},
\code{contrastProportions()}, or \code{posthocProportions()})
}
\description{
The function \code{anopa()} performs an ANOPA for designs with up to 4 factors
according to the ANOPA framework. See \insertCite{lc23;textual}{ANOPA} for more.
}
\details{
Note the following limitations:
\enumerate{
\item The main analysis performed by \code{anopa()} is currently restricted to three
factors in total (between and/or within). Contact the author if you plan to analyze
more complex designs.
\item If you have repeated-measure design, the data \emph{must} be provided in wide or
long format. The correlation between successes cannot be assessed once the data are
in a compiled format.
\item The data can be given in three formats:
\itemize{
\item \code{wide}: In the wide format, there is one line for each participant, and
one column for each between-subject factors in the design. In the column(s), the level
of the factor is given (as a number, a string, or a factor). For within-subject
factors, the columns contains 0 or 1 based on the status of the measurement.
\item \code{long}: In the long format, there is an identifier column for each participant,
a factor column and a level number for that factor. If there are n participants
and m factors, there will be in total n x m lines.
\item \code{compiled}: In the compiled format, there are as many lines as there are cells in the
design. If there are two factors, with two levels each, there will be 4 lines.
}
}

See the vignette \code{vignette("B-DataFormatsForProportions", package = "ANOPA")}
for more on data format and how to write their formula.
}
\examples{
# -- FIRST EXAMPLE --
# Basic example using a single between-subject factor design with the data in compiled format. 
# Ficticious data present success (1) or failure (0) of the observation according
# to the state of residency (three levels: Florida, Kentucky or Montana) for 
# 3 possible cells. There are 175 observations (with unequal n, Montana having only)
# 45 observations). 
minimalBSExample
# The data are in compiled format, consequently the data frame has only three lines.
# The complete data frame in wide format would be composed of 175 lines, one per participant.

# The following formula using curly braces is describing this data format
# (note the semicolon to separate the number of successes from the number of observations):
formula <- {s; n} ~ state

# The analysis is performed using the function `anopa()` with a formula and data:
w <- anopa(formula, minimalBSExample) 
summary(w)
# As seen, the proportions of success do not differ across states.

# To see the proportions when the data is in compiled format, simply divide the 
# number of success (s) by the total number of observations (n):
minimalBSExample$s / minimalBSExample$n

# A plot of the proportions with error bars (default 95\% confidence intervals) is
# easily obtained with
anopaPlot(w)

# The data can be re-formated into different formats with, 
# e.g., `toRaw()`, `toLong()`, `toWide()`
head(toWide(w))
# In this format, only 1s and 0s are shown, one participant per line.
# See the vignette `DataFormatsForFrequencies` for more.

# -- SECOND EXAMPLE --
# Real-data example using a three-factor design with the data in compiled format:
ArringtonEtAl2002

#  This dataset, shown in compiled format, has three cells missing 
# (e.g., fishes whose location is African, are Detrivore, feeding Nocturnally)
w <- anopa( {s;n} ~ Location * Trophism * Diel, ArringtonEtAl2002 )

# The function `anopa()` generates the missing cells with 0 success over 0 observations.
# Afterwards, cells with missing values are imputed  based on the option:
getOption("ANOPA.zeros")
# where 0.05 is 1/20 of a success over one observations (arcsine transforms allows 
# fractions of success; it remains to be studied what imputation strategy is best...)

# The analysis suggests a main effect of Trophism (type of food ingested)
# but the interaction Trophism by Diel (moment of feeding) is not to be neglected...
summary(w) # or summarize(w)

# The above presents both the uncorrected statistics as well as the corrected
# ones for small samples (Williams, 1976). You can obtain only the uncorrected...
uncorrected(w)

#... or the corrected ones
corrected(w)

# Finally, the data may have repeated measures and still be accessible in a compiled 
# format, as is the case of this short example:
minimalMxExampleCompiled

# As seen, it has one "group" factor (between) and two repeated measures (under the
# "foraging" or "frg" within factor). The groups are unequal, ranging form 16 to 81. 
# Finally, as this is repeated measures, there are correlations in each group
# (generally weak except possibly for the "treatment3" group).

# Such a compiled structure can be provided to anopa() by specifying the 
# repeated measures first (within cbind()), next the number of observation column, 
# and finally, the column containing the measure of correlation (any names can be used):
v <- anopa( {cbind(frg.before,frg.after); Count; uAlpha} ~ group, 
             minimalMxExampleCompiled,
             WSFactors = "foraging(2)")
anopaPlot(v)
summary(v)


# You can also ask easier outputs with:
explain(w)   # human-readable ouptut NOT YET DONE


}
\references{
\insertAllCited
}
